<?php

declare(strict_types=1);

namespace Mautic\PageBundle\Tests\Controller;

use Mautic\CoreBundle\Test\MauticMysqlTestCase;
use Mautic\DynamicContentBundle\Entity\DynamicContent;
use Mautic\LeadBundle\Entity\LeadList;
use Mautic\PageBundle\Entity\Page;
use Mautic\ProjectBundle\Entity\Project;
use PHPUnit\Framework\Assert;
use Symfony\Component\HttpFoundation\Request;

class PageControllerFunctionalTest extends MauticMysqlTestCase
{
    public function testPagePreview(): void
    {
        $segment = $this->createSegment();
        $filter  = [
            [
                'glue'     => 'and',
                'field'    => 'leadlist',
                'object'   => 'lead',
                'type'     => 'leadlist',
                'filter'   => [$segment->getId()],
                'display'  => null,
                'operator' => 'in',
            ],
        ];
        $dynamicContent = $this->createDynamicContentWithSegmentFilter($filter);

        $dynamicContentToken = sprintf('{dwc=%s}', $dynamicContent->getSlotName());
        $page                = $this->createPage($dynamicContentToken);

        $this->client->request(Request::METHOD_GET, sprintf('/%s', $page->getAlias()));
        $response = $this->client->getResponse();
        $this->assertSame(200, $response->getStatusCode());
        $this->assertStringContainsString('Test Html', $response->getContent());
    }

    private function createSegment(): LeadList
    {
        $segment = new LeadList();
        $segment->setName('Segment 1');
        $segment->setPublicName('Segment 1');
        $segment->setAlias('segment_1');
        $this->em->persist($segment);
        $this->em->flush();

        return $segment;
    }

    /**
     * @param mixed[] $filters
     */
    private function createDynamicContentWithSegmentFilter(array $filters = []): DynamicContent
    {
        $dynamicContent = new DynamicContent();
        $dynamicContent->setName('DC 1');
        $dynamicContent->setDescription('Customised value');
        $dynamicContent->setFilters($filters);
        $dynamicContent->setIsCampaignBased(false);
        $dynamicContent->setSlotName('Segment1_Slot');
        $this->em->persist($dynamicContent);
        $this->em->flush();

        return $dynamicContent;
    }

    private function createPage(string $token = ''): Page
    {
        $page = new Page();
        $page->setIsPublished(true);
        $page->setTitle('Page Title');
        $page->setAlias('page-alias');
        $page->setTemplate('blank');
        $page->setCustomHtml('Test Html'.$token);
        $this->em->persist($page);
        $this->em->flush();

        return $page;
    }

    public function testPageWithProject(): void
    {
        $page = $this->createPage();

        $project = new Project();
        $project->setName('Test Project');
        $this->em->persist($project);

        $this->em->flush();
        $this->em->clear();

        $crawler = $this->client->request('GET', '/s/pages/edit/'.$page->getId());
        $form    = $crawler->selectButton('Save')->form();
        $form['page[projects]']->setValue((string) $project->getId());

        $this->client->submit($form);

        $this->assertResponseIsSuccessful();

        $savedPage = $this->em->find(Page::class, $page->getId());
        $this->assertSame($project->getId(), $savedPage->getProjects()->first()->getId());
    }

    public function testPageWithNullCustomHtmlIsUpdated(): void
    {
        $page = new Page();

        $page->setTitle('Page A');
        $page->setAlias('page-a');
        $page->setTemplate('mautic_code_mode');

        $this->em->persist($page);
        $this->em->flush();

        $pageId        = $page->getId();
        $crawler       = $this->client->request(Request::METHOD_GET, '/s/pages/edit/'.$pageId);
        $buttonCrawler = $crawler->selectButton('Save & Close');
        $form          = $buttonCrawler->form();

        $form['page[title]']->setValue('New Page');

        $this->client->submit($form);

        $this->assertResponseIsSuccessful();

        $this->em->clear();

        Assert::assertEquals('New Page', $this->em->find(Page::class, $pageId)->getTitle());
    }
}
